'user sctrict'

function tokeniseExpression (expression) {
    if (!expression) return []
    const tokens = []
    let cursor = -1
    const length = expression.length
    function getNextToken () {
        while (true) {
            const nextC = expression[++cursor]
            if (nextC === ' ') {
            } else if (nextC === '+') {
                return ['op', 'plus']
            } else if (nextC === '-') {
                return ['op', 'minus']
            } else if (nextC === '*') {
                return ['op', 'multiply']
            } else if (nextC === '/') {
                return ['op', 'divide']
            } else if ((nextC >= '0' && nextC <= '9') || nextC === '.') {
                let currentToken = nextC
                while (true) {
                    if (++cursor >= length) return ['number', 'literal', parseFloat(currentToken)]
                    const nextC = expression[cursor]
                    if ((nextC >= '0' && nextC <= '9') || nextC === '.') {
                        currentToken += nextC
                    } else {
                        cursor--
                        return ['number', 'literal', parseFloat(currentToken)]
                    }
                }
            } else if ((nextC >= 'a' && nextC <= 'z') || (nextC >= 'A' && nextC <= 'Z')) {
                let currentToken = (nextC >= 'A' && nextC <= 'Z') ? String.fromCharCode(nextC.charCodeAt(0) + 32) : nextC
                while (true) {
                    if (++cursor >= length) return ['word', currentToken]
                    const nextC = expression[cursor]
                    if ((nextC >= 'a' && nextC <= 'z') || (nextC >= '0' && nextC <= '9')) {
                        currentToken += nextC
                    } else if ((nextC >= 'A' && nextC <= 'Z')) {
                        currentToken += String.fromCharCode(nextC.charCodeAt(0) + 32)
                    } else {
                        cursor--
                        return ['word', currentToken]
                    }
                }
            } else if (nextC === '"') {
                let currentToken = ''
                while (true) {
                    if (++cursor >= length) throw new Error('Expected terminating "')
                    const nextC = expression[cursor]
                    if (nextC === '"') {
                        return ['string', currentToken]
                    } else {
                        currentToken += nextC
                    }
                }
            } else if (nextC === '(') {
                return ['lparen']
            } else if (nextC === ')') {
                return ['rparen']
            } else if (nextC === ',') {
                return ['comma']
            } else if (nextC === undefined) {
                return undefined
            } else {
                throw new Error('Unexpected char in tokeniser')
            }
        }
    }
    while (true) {
        const token = getNextToken()
        if (token) {
            tokens.push(token)
        } else {
            return tokens
        }
    }
}

function buildTree (tokens, constants = {}) {
    if (tokens.length === 0) return undefined
    let tokenCursor = 0
    function getToken () {
        return tokens[tokenCursor++]
    }
    function peekToken () {
        return tokens[tokenCursor]
    }

    const bindTable = {
        plus: 1,
        minus: 1,
        multiply: 2,
        divide: 2,
    }

    function parse (bindLimit) {
        let left = getToken()
        if (left[0] === 'op' && left[1] === 'minus') {
            // Negation operator.
            left = ['multiply', ['number', 'literal', -1], parse(1)]
        } else if (left[0] === 'word') {
            const nextToken = peekToken()
            if (nextToken && nextToken[0] === 'lparen') {
                // Functions.
                getToken()
                const functionName = left[1]
                const args = []
                while (true) {
                    const nextToken = peekToken()
                    if (!nextToken) throw new Error('Expected rparen')
                    else if (nextToken[0] === 'rparen') {
                        getToken()
                        break
                    } else {
                        args.push(parse(0))
                        const nextToken = peekToken()
                        if (nextToken && nextToken[0] === 'comma') getToken()
                    }
                }
                left = ['function', functionName, args]
            } else {
                // Promote words to constants or variables.
                const variableName = left[1]
                const constantValue = constants[variableName]
                if (constantValue === undefined) {
                    // Variable value.
                    left = ['number', 'variable', variableName ]
                } else {
                    // Constant value.
                    left = ['number', 'literal', constantValue ]
                }
            }
        } else if (left[0] === 'lparen') {
            // Parenthesis.
            left = parse(0)
            const nextToken = getToken()
            if (!nextToken || nextToken[0] !== 'rparen') throw new Error('Expected rparen')
        }
        while (true) {
            const nextToken = peekToken()
            if (nextToken === undefined) return left
            const nextTokenType = nextToken[0]
            if (nextTokenType === 'rparen') return left
            else if (nextTokenType === 'comma') return left
            else if (nextTokenType !== 'op') throw new Error('Expected op')
            const opType = nextToken[1]
            const bind = bindTable[opType] ?? -1

            if (bind <= bindLimit) return left
            getToken()
            const right = parse(bind)
            left = [opType, left, right]
        }
    }

    return parse(0)
}


function stringifyTree (tree) {
    // console.log(JSON.stringify(tree))
    let result = ''
    function recurse (node) {
        const type = node[0]
        if (type === 'number') {
            result += node[2]
        } else if (type === 'function') {
            result += node[1] + '( '
            let argCount = node[2].length
            node[2].forEach(arg => {
                recurse(arg)
                if (--argCount > 0) result += ', '
            })
            result += ')'
        } else if (type === 'plus') {
            result += '( '
            recurse(node[1])
            result += '+ '
            recurse(node[2])
            result += ')'
        } else if (type === 'minus') {
            result += '( '
            recurse(node[1])
            result += '- '
            recurse(node[2])
            result += ')'
        } else if (type === 'multiply') {
            result += '( '
            recurse(node[1])
            result += '* '
            recurse(node[2])
            result += ')'
        } else if (type === 'divide') {
            result += '( '
            recurse(node[1])
            result += '/ '
            recurse(node[2])
            result += ')'
        } else if (type === 'string') {
            result += `"${node[1]}"`
        } else {
            result += '???'
        }
        result += ' '
    }
    recurse(tree)
    return result
}

function calculateTree (tree, variables = {}, functions = {}) {
    function recurse (node) {
        const type = node[0]
        if (type === 'number') {
            const numberType = node[1]
            if (numberType === 'literal') return node[2]
            else if (numberType === 'variable') {
                const result = variables[node[2]]
                if (result === undefined) throw new Error('Unknown variable')
                return result
            }
            // error
        } else if (type === 'plus') {
            return recurse(node[1]) + recurse(node[2])
        } else if (type === 'minus') {
            return recurse(node[1]) - recurse(node[2])
        } else if (type === 'multiply') {
            return recurse(node[1]) * recurse(node[2])
        } else if (type === 'divide') {
            return recurse(node[1]) / recurse(node[2])
        } else if (type === 'function') {
            const result = functions[node[1]]
            if (result === undefined) throw new Error('Unknown function')
            const args = node[2].map(arg => {
                if (arg[0] === 'string') {
                    return arg[1]
                } else {
                    return recurse(arg)
                }
            })
            return result(args)
        } else {
            throw new Error('Unexpected token')
        }
    }
    return recurse(tree)

}

class ExpressionParserCache {
    constructor (constants = {}, functions = {}) {
        this.constants = constants
        this.functions = functions
        this.expressionCache = {}
    }

    executeExpression (expression, variables = {}) {
        // Attempt to build tree in cache if not already there.
        if (!this.expressionCache[expression]) {
            try {
                const tokens = tokeniseExpression(expression)
                const tree = buildTree(tokens, this.constants)
                this.expressionCache[expression] = tree
            } catch (e) {
                return undefined
            }
        }

        // Execute tree from cache.
        try {
            const tree = this.expressionCache[expression]
            const result = calculateTree(tree, variables, this.functions)
            return result
        } catch (e) {
            return undefined
        }
    }
}

try {
    module.exports = {
        tokeniseExpression,
        buildTree,
        stringifyTree,
        calculateTree,
    }
} catch (e) {
}
